//CommonDimensions.scad;

//Common dimensions for case for 
//Noughts and Crosses using Arduino;

//This is version 2022_01_12;
//It can produce a case with 
//WallInnerX = 80, for Adafruit V2 
//touch screens, or a case 
//WallInnerX = 86, for clones based on 
//the Adafruit V1 touch screen;

//The selection is clumsy:
//Comment one and only one of 
//the assignments below;
//WallInnerX = 80;
WallInnerX = 86;

//It is probably a general purpose case 
//for many projects using a UNO and a TFT;

//The floor is in two parts:
//A lower floor clamps an upper floor 
//between itself and the lugs on the lid,
//but does not hold the UNO;
//The upper floor has the posts 
//to secure the UNO;
//These posts can be drilled 
//and tapped right thru;

//The preceeding designs needed the threads 
//in the posts to be bottom tapped, 
//and that is unnecessarily difficult;

//Use an include statement to include 
//these definitions in appropriate 
//modules;

//Most dimensions are relatively obvious;

//Early prototypes defined most 
//offsets relative to the X and Y axes;
//Although this seemed like a good idea, 
//it caused many problems, 
//and circa 2021_12_06 all modules 
//were revised to make all dimensions 
//relative to the origin;
//That has benefits and liabilities, 
//but I seem to win more on the swings 
//than I lose on the roundabouts;

//I'll make my case with a relatively 
//shallow floor and a correspondingly 
//deep lid;

//The lid will enclose the floor;
//An earlier design had the lid resting on 
//a rebate around the edges of the floor, 
//and that became complicated;

//I'll leave a hole in the lid at the 
//left for the USB connector and another 
//for the DC power connector;

Fuzz=0.010;

Whisker = 1;

WallThickness = 3;  //Was WallBottom;

//The UNO and the TFT attached to it have 
//some symmetry and all four obvious 
//rotations can be accommodated by 
//modifying the software appropriately; 
//The UNO does not have a sensor to 
//automate this rotation;

//The dimensions below provide ample room 
//for the walls of the lid to fit around 
//the UNO and TFT;

//When the walls enclose the floor, 
//the benchmark dimensions are not FloorX 
//and FloorY, but WallInnerX and WallInnerY;

//WallInnerX is defined earlier;
//Don't define it here;
//WallInnerX = nn;
WallInnerY = 70;
WallInnerZ = 22;

FloorX = WallInnerX -Whisker;   
FloorY = WallInnerY -Whisker;    
FloorZ = 4;

//The case must include: 
//WallInnerZ, two floors, and one top;
WallZ = WallInnerZ +2*FloorZ +WallThickness;

//The holes for the USB socket and for the 
//DC power plug are in the bottom of the 
//wall of the lid at the left side of the UNO;

//Both holes are simple rectangles;
//They are constructed at the origin in 
//their appropriate orientation and only need 
//to be translated to their 
//appropriate positions;

//The hole for the USB;

//The hole is just a hole,
//and can be of generous thickness;
USBx = 2*WallThickness;
USBy = 14;
USBz = 17 +2*FloorZ;
    
//Its location is a "Select On Test" value,
//found by trial and error,
//including quite a lot of error;
//The values below use a filler 
//exactly WallThickness thick;
//See: USBfillerX = WallThickness;
//It doesn't align perfectly with the wall;
//Probably, I should make it a little 
//thicker and use -WallThickness instead 
//of -WallThickness/2 for DCdeltaX;
//Probably, maybe, but not today;
USBdeltaX = -WallInnerX/2 
            -USBx/2
            +WallThickness/2;
USBdeltaY = 12; //Was 11;
USBdeltaZ = USBz/2 -Fuzz;
    
//The hole for the DC power plug;

DCx = USBx;
DCy = 11;
DCz = USBz;
    
//Its location is a "Select On Test" value,
//found by trial and error,
//including quite a lot of error;
//The values below use a filler 
//exactly WallThickness thick;
//See: DCfillerX = WallThickness;
//It doesn't align perfectly with the wall;
//Probably, I should make it a little 
//thicker and use -WallThickness instead 
//of -WallThickness/2 for DCdeltaX;
//Probably, maybe, but not today;
DCdeltaX = USBdeltaX;
DCdeltaY = -19; //Was -19, then -18;
DCdeltaZ = DCz/2 -Fuzz;

//Measurement indicates that 
//USBdeltaY = DCdeltaY +30,
//even though USBdeltaY -DCdeltaY = 29;

//The gap below the USB connector needs to be 
//filled by a piece of wall attached to 
//the floor; 

//The filler for the USB connector;

//The depth of the filler is the thickness 
//of the wall of the case;
USBfillerX = WallThickness;

//I considered making 
//USBfillerX = 2*WallThickness
//and adjusting USBfillerDeltaX accordingly;
//That would make the join between the 
//filler and its floor a little stronger, 
//but it would also need recesses in the 
//floor above;
//The join is already quite strong, 
//and I deemed its strength to be adequate;
    
//The width of the filler is a whisker less 
//than the width of the hole in the lid, 
//but not necessarily less than the width 
//of the USB connector;

USBfillerY = USBy -Whisker;

//The height is from the bottom of the floor 
//to the bottom of the USB connector;
USBfillerZ = 5 +2*FloorZ;

//The filler is constructed at the origin;
USBfillerDeltaX = -WallInnerX/2 
                    -USBfillerX/2 
                    +Whisker/2 +Fuzz;
USBfillerDeltaY = USBdeltaY;
USBfillerDeltaZ = USBfillerZ/2;

//The filler for the DC power connector;

//The depth of the filler is the thickness 
//of the wall of the case;
DCfillerX = WallThickness;

//I considered making 
//DCfillerX = 2*WallThickness
//and adjusting DCfillerDeltaX accordingly;
//That would make the join between the 
//filler and its floor a little stronger, 
//but it would also need recesses in the 
//floor above;
//The join is already quite strong, 
//and I deemed its strength to be adequate;
    
//The width of the filler is a whisker less 
//than the width of the hole in the lid, 
//but not necessarily less than the width 
//of the DC power connector;
DCfillerY = DCy -Whisker;

//The height is from the bottom of the floor 
//to the bottom of the power connector;
DCfillerZ = USBfillerZ;

//The filler is constructed at the origin;
DCfillerDeltaX = -WallInnerX/2 
                    -DCfillerX/2 
                    +Whisker/2 +Fuzz;
DCfillerDeltaY = DCdeltaY;
DCfillerDeltaZ = DCfillerZ/2;

//I will name the mounting posts mnemonically:
//BL = Bottom Left, TR = Top Right, etc;  

//The mounting holes seem to be on 
//an imperial grid;
//I've calculated each conversion 
//to mm explicitly;  

//Measurements indicate that the 
//BR and TR holes might be on an 
//imperial grid while the 
//BL and TL holes are on a metric grid;

//Mounting posts for UNO;
//Current values for 
//WallInnerX = 86;
//WallInnerY = 70;

//Warning!
//The defined dimension is the distance 
//from WallInnerX/2 to the post, 
//not the distance from the origin to the post;
PostUnoBLx = -(WallInnerX/2 -21);    
PostUnoBLy = -24;     

PostUnoBRx = PostUnoBLx +52.5;    
PostUnoBRy = PostUnoBLy +5;    

PostUnoTRx = PostUnoBRx;        
PostUnoTRy = PostUnoBLy +33;    

PostUnoTLx = PostUnoBLx +1.5; 
PostUnoTLy = PostUnoBLy +49;    

//The mounting post for the UNO is 
//a simple cone with a hole in the middle;
PostUnoPhiBottom = 6;
PostUnoPhiTop = 4;
PostUnoHeight = 4;

//A chart of drill sizes 
//for tapped holes indicates:
//Tap mm    Drill mm;
//M2        1.6;
//M2.5      2.05;
//M3        2.5;
//M4        3.3;
//M5        4.2;
//M6        5.0;

//For this version, 
//the holes in the BL, BR, TR mounting posts 
//are PostUnoPhiBigTap = 2.5mm, 
//ready to be tapped for a 3mm screw;
//The hole in the TL mounting post 
//is PostUnoPhiSmallTap = 2.0mm, 
//ready to be tapped for a 2.5mm screw;
PostUnoPhiBigTap = 2.5;
PostUnoPhiSmallTap = 2.0;

//I need something to hold 
//the lid to the floor;
//After quite a lot of thought, 
//I decided to attach pillars to the walls 
//of the lid, put tapped holes in them, 
//and have holes in the floor for screws;  
//I'd rather keep the floor smooth and 
//clear, but I can't think of anything 
//better than holes at this time;
//The diameter of the thread is 
//a bit of a surprise;
//The screws are sold as 3mm;
//I thought I'd need a hole bigger than 3mm;
//It seems that I don't;

PostLidPhi = 5;
PostLidPhiBigTap = 2.5;
PostLidHeight = 6;

LidScrewHeadPhi = 6;
LidScrewHeadDepth = 2;
LidScrewThreadPhi = 3.2;

//Mounting posts for the lid;
//Current values for translate([])
//for FloorY = 76;

//Provided I use the same values for 
//the screw holes in the floor and the
//pillars on the walls of the lid, 
//the components should align correctly;

//The values below have been adopted 
//circa 2021_11_22, after quite a lot 
//of trial and error, especially error;  
//Earlier trials put the pillars 
//too close to the PCBs;

LidDeltaX = (WallInnerX/2 -15);

LidBLx = -LidDeltaX;    
LidBLy = -30;     

LidBRx = LidBLx +55;    
LidBRy = LidBLy;    

LidTRx = LidBRx;        
LidTRy = -LidBLy;    

LidTLx = LidBLx; 
LidTLy = -LidBLy;    

//I want to use two magnets to attach 
//the battery case to the UNO case;
//In particular: Jaycar LM1622;
//These are 10mm Phi, 3mm high;
//I want most of the hole for them in 
//FloorOne and a smaller portion of the 
//hole for them in FloorTwo;
//However, if the holes are the same size, 
//the magnet will naturally position 
//itself into FloorOne;
MagnetHolePhi = 11;
MagnetHoleHeight = 2;

//I need to define the translations 
//for the holes globally, so all holes 
//and magnets align properly;
//I'll place the magnets on the centerlines 
//of the floors, so one of MagnetHoleDeltaX 
//or MagnetHoleDeltaY will be zero; 

//Although relative definitions look 
//elegant, to ensure that the magnets in 
//the case for the UNO align with the 
//magnets in the case for the battery, 
//it is safer to define MagnetHoleDeltaX 
//and MagnetHoleDeltaY absolutely;
//MagnetHoleDeltaX = 55/2;
//MagnetHoleDeltaY = 0;
MagnetHoleDeltaX = 0;
MagnetHoleDeltaY = 25;

//A subset of a chart of drill sizes 
//for tapped holes:
//Tap mm    Drill mm;
//M2        1.6;
//M2.5      2.05;
//M3        2.5;
//M4        3.3;
//M5        4.2;
//M6        5.0;

//An attractive subset of the names 
//OpenSCAD recognizes for colours;
//I can remember "blue", but continually 
//forget the names of the others;
//Reds
//  LightCoral
//  LightSalmon
//Oranges
//  Coral
//  Orange
//Yellows
//  LemonChiffon
//Greens
//  LightGreen
//  YellowGreen
//Blues
//SkyBlue
//  CornflowerBlue
//Violets
//  Plumb
//  Violet
//Browns
//  Tan

//
//
